# /// script
# requires-python = ">=3.13"
# dependencies = [
#     "kaleido==1.2.0",
#     "marimo",
#     "matplotlib==3.10.8",
#     "numpy==2.4.1",
#     "pandas==3.0.0",
#     "plotly==6.5.2",
#     "scipy==1.17.0",
# ]
# ///

import marimo

__generated_with = "0.19.6"
app = marimo.App(width="medium")


@app.cell(hide_code=True)
def _(mo):
    mo.hstack(
        [
            mo.image(
                src="http://modernmacro.org/resources/Vivaldo/figures/OldTests/RIR.png",
                width=200,
            ),
            mo.image(
                src="http://modernmacro.org/resources/Vivaldo/figures/iscte.png",
                width=300,
            ),
        ],
        align="center",
    )
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    # Day 1 - Welcome to Marimo Notebooks

    ## Introduction to Marimo
    <br>
    **Vivaldo Mendes**

    **February 02, 2026**
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    <style>
    .marimo-cell {
        border: none !important;
        box-shadow: none !important;
        margin: 0 !important;
        padding: 0 !important;
    }

    .marimo-cell-content {
        padding: 0.5rem 0 !important;
    }
    .admonition {
    margin-top: -0.99rem !important;
    margin-bottom: -0.99rem !important;
    }
    h1 {
        margin-bottom: -1.2rem !important;
    }
    h2 {
        margin-bottom: -1.2rem !important;
    }
    /* Reduce spacing before bullet lists */
    ul {
        margin-top: -0.3rem !important;
    }
    /* Optionally reduce spacing between list items */
    li {
        margin-bottom: 0.2rem !important;
    }
    /* Increase font size for all text in the document */
    body {
        font-size: 16px;
    }

    /* Increase font size for all headers */
    h1 {
        font-size: 36px !important;
    }

    h2 {
        font-size: 28px !important;
    }

    h3 {
        font-size: 25px !important;
    }
    </style>
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ---
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ## Libraries used in this notebook
    """)
    return


@app.cell
def _():
    import marimo as mo                                    # Reactive Python notebooks
    import numpy as np                                     # Numerical computing
    import pandas as pd                                    # Data manipulation
    from scipy.optimize import fsolve                      # Scientific computing
    import plotly.graph_objects as go                      # Interactive plotting
    from plotly.subplots import make_subplots
    import plotly.express as px
    import plotly.io as pio 
    pio.templates.default = "plotly"
    import kaleido
    pio.defaults.default_format = "svg"
    from datetime import datetime
    import warnings
    warnings.filterwarnings('ignore')
    import os
    os.chdir(os.path.dirname(__file__))
    return fsolve, go, mo, np, os, pd, px


@app.cell
def _(os):
    #import os
    os.getcwd()
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ---
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ## 1. What is a Marimo notebook?
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    A notebook is a document with two types of cells (see image below):
    - **Code cells** - contain Python code that can be executed
    - **Markdown cells** - contain formatted text, mathematics, images, and more

    The versatility that arises from combining markdown cells with code cells makes notebooks a remarkable tool for computational work in general and teaching in particular.

    Marimo notebooks are **reactive**: when you change a cell, all cells that depend on it automatically re-run. This ensures your notebook is always in a consistent state.
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.Html("""
    <div style="display: flex; justify-content: left; align-items: left;">
        <img 
    		src="http://modernmacro.org/resources/Vivaldo/figures/Marimo/marimo_cells.png" 
    	alt="Centered Image" style="max-width: 70%; height: auto;">
    </div>
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ---
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ## 2. Marimo basic tips
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ### Creating and running cells

    - Put the mouse on top of any opened/visible cell.
    - You will see a <kbd>+</kbd> sign in the top and bottom left corners (**_see image below_**).
    - Click on the <kbd>+</kbd> sign where you want the new cell to be placed.
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.Html("""
    <div style="display: flex; justify-content: left; align-items: left;">
        <img 
    		src="http://modernmacro.org/resources/Vivaldo/figures/Marimo/marimo_newcells.png" 
    	alt="Centered Image" style="max-width: 70%; height: auto;">
    </div>
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ### How to run a cell?

    - See **_image above_**
    - Click on the small circle-arrow in the top right-hand-side of each cell: <kbd>▶️</kbd>
    - Or, click simultaneosly on <kbd>Ctrl</kbd> <kbd>↩ Enter</kbd>
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ### How to save a notebook?

    - When a cell is run, this action will be automatically saved
    - If we forget to run a particular cell, click simultaneously on <kbd>Ctrl</kbd> <kbd>Shift</kbd> <kbd>R</kbd> to save all changes that may have been implemented and not yet saved. There is also a button to save all stale cells (cells not yet run) at the bottom right corner of the notebook: <kbd>▶</kbd>
    - If we want to save the notebook, independently of whether all the cells were run (or not), click simultaneously on <kbd>Ctrl</kbd> <kbd>S</kbd>; there is also a button for this task at the bottom right corner of the notebook: <kbd>💾</kbd>
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ### Code cells
    - When we click on the <kbd>+</kbd> sign, as shown above, a new cell will be automatically opened
    - By default, this new cell will be a code cell
    - The next cell is an empty code cell that can be used for computation. For example, type 2+2 and run it.
    """)
    return


@app.cell
def _():
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ### Markdown cells
    - If we want to write text or mathematics in a new cell, or insert a video, we have to turn it into a "markdown" cell
    - To change a cell from code mode into **_markdown mode_**, put the cursor inside the cell you want to be in markdown.
      - Click simultaneously on <kbd>Ctrl</kbd> <kbd>Shift</kbd> <kbd>M</kbd>
      - ... or you can click on the **button** that does that task (see image above): <kbd>Ⓜ️⬇</kbd>
    - That is all. You can start typing text or mathematics as you like.
    """)
    return


@app.cell
def _():
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    /// tip | Very common formats

    To write text in bold, italics, or mathematical symbols:
    - Type: `**This is bold**` to get: **This is bold**
    - Type: `*This is italics*` to get: *This is italics*
    - Type: `***This is bold and italics***` to get: ***This is bold and italics***
    - Type: `$y=2x^2$` to get a mathematical expression: $y=2x^2$
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    /// tip | Lists
    - This is my first item
    - This is my second item
      - Two spaces give my first sub-item
      - My second
        - Four spaces give first sub-sub-item
        - Very easy
    - My final item

    1. My first numbered item
    2. Wow, this is easy
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    /// tip | Mathematical text

    To write mathematical expressions in markdown, we use LaTeX syntax with `$`.

    This is inline mathematics: $y=2x$.

    This is displayed mathematics:

    $$y=2x$$

    This is mathematics using the power syntax: $y = 2x^3$

    This is mathematics using the fraction syntax: $h = \frac{3}{4n}$

    This is more elaborate mathematics: $z = \int_{a}^{b} x^2 dx$

    This is using Greek symbols and a tag (1):

    $$\lambda = \sum_{i=0}^{\infty} \alpha^i x_{i-1} \tag{1}$$
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    /// tip | Tables

    It is extremely easy to produce a table. Use the `|` to separate columns and `:` to align columns:

    | Model       | Prompt runs | Output parsable | Infrastructure | Result |
    |:------      |:-----------:|:---------------:|:--------------:|:------:|
    | GPT-4o     | ✓           | ✓               | ✓             | ✅ Works |
    | Gemini 2.0  | ✓           | ✓               | ⚠️ Budget     | ⚠️ Partial |
    | Gemma 2 27B | ✓           | ✗               | ✓             | ❌ Format |
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    👉 **Exercise 1**

    - Open a new cell. Write the following text: `Today is a sunny day!`
    - Open another cell and write the same text but in bold, and also in italics.
    - Open a new cell and write: This is text and mathematics with a funny equation $y = xz/2$
    - Open a new cell and write the following equation: $y = 3x^2$
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    /// hint | Answer: Exercise 1

    Today is a sunny day!

    **Today is a sunny day!**

    ***Today is a sunny day!***

    This is text and mathematics with a funny equation $y = xz/2$

    $y = 3x^2$
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ---
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ## 3. Arithmetic operators
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    /// danger | Python supports the following arithmetic operators:

    <kbd>+x</kbd> - unary plus (identity operation) <br>
    <kbd>-x</kbd> - unary minus (negation) <br>
    <kbd>x + y</kbd> - addition <br>
    <kbd>x - y </kbd> - subtraction <br>
    <kbd>x * y</kbd> - multiplication <br>
    <kbd>x / y</kbd>  - division (always returns float) <br>
    <kbd>x // y</kbd>  - floor division (integer division) <br>
    <kbd>x % y</kbd>  - modulus (remainder) <br>
    <kbd>x ** y</kbd>  - exponentiation (power)
    """)
    return


@app.cell
def _():
    2+3
    return


@app.cell
def _():
    10**4
    return


@app.cell
def _():
    pepe = 10
    return (pepe,)


@app.cell
def _():
    rock = 20
    return (rock,)


@app.cell
def _(pepe, rock):
    mary = pepe * rock
    mary
    return (mary,)


@app.cell(hide_code=True)
def _(mo):
    fiona_slider = mo.ui.slider(-5.0, 5.0, value=1.0, step=0.5, show_value=True, label="fiona")
    fiona_slider
    return (fiona_slider,)


@app.cell
def _(fiona_slider, mary):
    fiona = fiona_slider.value
    paty = (fiona * mary, fiona**2, fiona/mary, 10*fiona)
    paty
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    👉 **Exercise 2**

    What is the value of a variable called "zazu", given the following equation?

    $$zazu = 10 + rock^2 + \frac{pepe}{2} \tag{2}$$

    Use the cell below to obtain the solution to equation (2).
    """)
    return


@app.cell
def _(pepe, rock):
    # Answer to Exercise 2
    zazu = 10 + rock**2 + pepe/2
    zazu
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    /// hint | Answer: Exercise 2

    zazu = 415
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ---
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ## 4. Plotting functions
    """)
    return


@app.cell
def _(np):
    x = np.arange(0.0, 20.0, 0.05)
    Ana = 2 * np.sin(x)
    return Ana, x


@app.cell
def _(Ana, px, x):
    # Simple sine wave plot
    fig1 = px.line(x=x, y=Ana)
    fig1.update_layout(height=450, hovermode='x', title_text='My fancy plot', title_x=0.5, xaxis_title = "x", yaxis_title = "Ana", margin=dict(l=70, r=60, t=70, b=60))
    fig1
    return


@app.cell
def _(np):
    noise = np.random.randn(500)
    return (noise,)


@app.cell
def _(go, noise):
    # Histogram of random noise
    fig3 = go.Figure()
    fig3.add_trace(go.Histogram(x=noise, nbinsx=50, opacity=0.75))
    fig3.update_layout(bargap=0.05, height=450, margin=dict(l=70, r=60, t=70, b=60))
    fig3
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    👉 **Exercise 3**

    In the following cell, produce a simple plot of the following function: $e^{Ana}$
    """)
    return


@app.cell
def _(Ana, go, np, x):
    # Exercise 3 answer
    fig_exp = go.Figure()
    fig_exp.add_trace(go.Scatter(x=x, y=np.exp(Ana), mode='lines'))
    fig_exp.update_layout(height=450, margin=dict(l=70, r=60, t=70, b=60))
    fig_exp
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    👉 **Exercise 4**

    In the next cell, we ***hide the code*** that plots three functions together. To run the cell, click the blurred area of the cell to bring its contents back to the surface. To display its output (a plot with three functions), delete the semi-comma <kbd>;</kbd> that can be ound in the last line, and finally run the cell (<kbd>Shift</kbd> and <kbd>Enter</kbd>, or click on the <kbd>▶️</kbd> button). The plot will then be produced.
    """)
    return


@app.cell(hide_code=True)
def _(go, np):
    # Multiple traces with different styles
    x2 = np.linspace(0, 3*np.pi, 100)

    fig4 = go.Figure()
    fig4.add_trace(go.Scatter(
        x=x2, y=np.cos(x2), mode='lines',
        line=dict(width=2, color='blue'),
        name='Shrek'
    ))
    fig4.add_trace(go.Scatter(
        x=x2, y=np.sin(x2), mode='lines',
        line=dict(dash='dashdot', color='BlueViolet'),
        name='Fiona'
    ))
    fig4.add_trace(go.Scatter(
        x=x2, y=np.cos(x2)**2 - 1/2, mode='markers+lines',
        marker=dict(symbol='circle-open', size=6, color='red'),
        line=dict(width=0.5),
        name='Donkey'
    ))
    fig4.update_layout(
        height=450,
        hovermode='x',
        title_text='My fancy plot with different lines & marker styles',
        title_x=0.5,
        xaxis_range = [-0.1, 9.5],
        margin=dict(l=70, r=60, t=70, b=60)
    )
    fig4
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ---
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ## 5. Magic things we can do with Marimo
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ### Write normal text and sophisticated mathematics
    Consider now an optimization problem where the constraints are defined in terms of inequalities. This can be solved by using the famous Karush-Kuhn–Tucker conditions, which require too much space to be explained here. An excellent place to look at is Chapter 6 of the textbook: Rangarajan K. Sundaram (1996). *A First Course in Optimization Theory*, Cambridge University Press.

    $$\max \quad x_{1}+2 x_{2}+5 x_{3}$$

    subject to

    $$\begin{aligned}
    -x_{1}+x_{2}+3 x_{3} & \leq-5 \\
    x_{1}+3 x_{2}-7 x_{3} & \leq 10 \\
    0 \leq x_{1} & \leq 10 \\
    x_{2} & \geq 0 \\
    x_{3} & \geq 0
    \end{aligned}$$

    And then, blah blah blah ....
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ---
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ### Use Unicode symbols for writing text and doing computation

    Contary to Julia, in Python, we ***cannot*** use fancy unicode characters to do computation. I have a 😺 that loves the 🌚, but not 🍍. In Julia, we can write simple text or do computations with these adorable objects. In Python, we can use them only as part of markdown text.

    To write special characters that are frequently used in Economics (like greek letters:  β, α, for example) in Python, you have to copy-paste from character lists and such an exercise is not very efficient:
    - Copy+paste unicode directly from a list: β, α, θ , φ

    - Complete list: α β γ δ ε ζ η θ ι κ λ μ ν ξ ο π ρ σ τ υ φ χ ψ ω
    Α Β Γ Δ Ε Ζ Η Θ Ι Κ Λ Μ Ν Ξ Ο Π Ρ Σ Τ Υ Φ Χ Ψ Ω
    """)
    return


@app.cell
def _():
    β = 3
    return (β,)


@app.cell
def _(β):
    α = 10 * β
    return (α,)


@app.cell
def _(β):
    beta_squared = β**2
    return (beta_squared,)


@app.cell
def _(beta_squared, α, β):

    β, α, beta_squared
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ---
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ### Play with sliders
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    👉 **Exercise 5**

    Move the two sliders above (one slide at a time), and see what happens to the equilibrium in the plot above.

    **Answer:** The equilibrium will shift based on changes in aggregate demand (ΔĀ) and potential output (ΔYᴾ).
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    # Supply and demand sliders
    delta_A_slider = mo.ui.slider(-1.0, 1.0, value=0.0, step=0.01, show_value=True, label="ΔĀ")
    delta_Yp_slider = mo.ui.slider(-1.0, 1.0, value=0.0, step=0.05, show_value=True, label="ΔYᴾ")

    mo.hstack([delta_A_slider, delta_Yp_slider],justify="center", gap=6)
    return delta_A_slider, delta_Yp_slider


@app.cell(hide_code=True)
def _(delta_A_slider, delta_Yp_slider, go, np):
    # Supply and demand equilibrium plot
    delta_A = delta_A_slider.value
    delta_Yp = delta_Yp_slider.value

    # Parameters
    A_bar = 8.0
    lambda_param = 0.5
    m = 2.0
    phi = 0.2
    r_bar = 2.0
    gamma = 2.8
    Yp = 14.8
    pi_e = 2.0
    rho = 0.0

    Y_range = np.arange(13.8, 15.6, 0.01)

    # AD curve
    pi_d_max = (A_bar / (lambda_param * phi)) - r_bar / lambda_param
    pi_d = pi_d_max - (1 / (m * phi * lambda_param)) * Y_range

    # AS curve
    pi_s_min = pi_e - gamma * Yp + rho
    pi_s = pi_s_min + gamma * Y_range

    # Updated AD curve with delta_A
    pi_d_max_new = ((A_bar + delta_A) / (lambda_param * phi)) - r_bar / lambda_param
    pi_d_new = pi_d_max_new - (1 / (m * phi * lambda_param)) * Y_range

    fig_econ = go.Figure()

    # Potential output line
    fig_econ.add_trace(go.Scatter(
        x=[Yp + delta_Yp, Yp + delta_Yp], y=[-5, 10],
        mode='lines', line=dict(width=4, color='Orange'),
        name='Yᴾ'
    ))

    # Original AD and AS curves
    fig_econ.add_trace(go.Scatter(x=Y_range, y=pi_d, mode='lines', 
                                    line=dict(color='Blue', width=3), name='AD'))
    fig_econ.add_trace(go.Scatter(x=Y_range, y=pi_s, mode='lines',
                                    line=dict(color='Red', width=3), name='AS'))

    # New AD curve
    if delta_A != 0:
        fig_econ.add_trace(go.Scatter(x=Y_range, y=pi_d_new, mode='lines',
                                        line=dict(color='Magenta', width=3), name='AD2'))

    # Equilibrium point
    fig_econ.add_trace(go.Scatter(
        x=[14.8], y=[2.0], mode='markers+text',
        marker=dict(size=12, color='Blue'),
        text=['1'], textposition='top center',
        textfont=dict(size=18, color='black'),
        name='Eq1', showlegend=False
    ))

    fig_econ.update_layout(
        height=450,
        title_text='Initial long-term equilibrium',
        title_x=0.5,
        hovermode='x',
        xaxis=dict(title='GDP trillion dollars (Y)', range=[14.2, 15.4]),
        yaxis=dict(title='Rate of inflation (π)', range=[-0.8, 5]),
        margin=dict(l=70, r=60, t=70, b=60)
    )

    fig_econ
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    /// hint | Answer: Exercise 5

    The equilibrium initially given by point 1 will cahnge.
    - ΔĀ will force the AD curve to shift
    - ΔYᴾ will force the Yᴾ curve to shift
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    👉 **Exercise 6**

    Move the two sliders above (one slide at a time), and see what happens to the sinusoidal wave.
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    # Wave sliders
    frequency_slider = mo.ui.slider(0.0, 3.0, value=1.0, step=0.1, show_value=True, label="frequency")
    damping_slider = mo.ui.slider(-0.5, 0.8, value=0.0, step=0.01, show_value=True, label="damping")

    mo.hstack([frequency_slider, damping_slider], justify="center", gap=6)
    return damping_slider, frequency_slider


@app.cell(hide_code=True)
def _(damping_slider, frequency_slider, go, np):
    # Damped sinusoidal wave
    frequency = frequency_slider.value
    damping = damping_slider.value

    tau = np.arange(1, 20, 0.01)
    wave = np.exp(-damping * tau) * np.sin(frequency * tau)

    fig_wave = go.Figure()
    fig_wave.add_trace(go.Scatter(x=tau, y=wave, mode='lines'))
    fig_wave.update_layout(height=450, margin=dict(l=70, r=60, t=70, b=60))
    fig_wave
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ### Calculate an integral
    👉 **Exercise 7**

    We can estimate the integral of a function $F:[a,b] \rightarrow \mathbb{R}$ using the midpoint rule approximation. Let us interactively estimate $\int_0^1 F(x) dx$ in the case of:

    $$F(x)= - x\sin(2\pi x)$$
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    Nbins_slider = mo.ui.slider(1, 500, value=20, step=1, show_value=True, label="Number of intervals")
    Nbins_slider
    return (Nbins_slider,)


@app.cell(hide_code=True)
def _(Nbins_slider, go, np):
    # Midpoint rule integral approximation
    Nbins = Nbins_slider.value

    def F_func(x):
        return -x * np.sin(2 * np.pi * x)

    centres = np.arange(1/(2*Nbins), 1, 1/Nbins)
    midpointarea = round(sum(F_func(centres)) / Nbins, 6)

    fig_integral = go.Figure()
    fig_integral.add_trace(go.Bar(x=centres, y=F_func(centres), name='Area'))
    fig_integral.add_trace(go.Scatter(x=centres, y=F_func(centres), mode='lines', name='F(x)'))
    fig_integral.update_layout(
        height=450,
        title_text=f'Midpoint rule estimate with {Nbins} intervals = {midpointarea}',
        margin=dict(l=70, r=60, t=70, b=60)
    )
    fig_integral
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ---
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ### Playing with questions & answers
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    👉👉 **Exercise 8**

    Table 1 shows some important macroeconomic aggregates, measured in **real terms**, for the US economy in 2013. Answer questions **a)** and **b)** below.

    | Code | Headings | Billions of dollars |
    |------|:----------|---------------------|
    | cons | Private Consumption Expenditures | 11400 |
    | inve | Investment in Equipment and Structures | 2600 |
    | stoc | Inventories | 80 |
    | gove | Public Expenditures | 3120 |
    | expo | Exports | 2250 |
    | impo | Imports | 3000 |
    """)
    return


@app.cell
def _():
    # Values from Table 1 passed into the notebook
    cons = 11400
    inve = 2600
    stoc = 80
    gove = 3120
    expo = 2250
    impo = 3000
    return cons, expo, gove, impo, inve, stoc


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    👉 **a)** Using Table 1, calculate GDP by the expenditure method.

    /// warning | Hint
    According to the usual definition of GDP by the expenditure method, GDP is equal to the sum of private consumption, plus public expenditure on goods and services, plus investment and inventories, plus exports minus imports.
    """)
    return


@app.cell
def _(cons, expo, gove, impo, inve, stoc):
    # Calculate GDP
    GDP = cons + inve + stoc + gove + expo - impo
    GDP
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    /// hint | Answer (a)

    GDP = 16450 billion dollars
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    👉 **b)** In a market economy, such as the one we live in, do you have any suggestion about the main macroeconomic variables that explain the behavior of private investment?

    /// hint |**Answer (b):**

    We have not yet discussed the main determinants of private investment in our economies. As we will see later in this course, those determinants are: real interest rates, financial frictions, and autonomous investment decisions.
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    👉 **Exercise 8A**  <br>

    We can use Marimo to create multiple-choice questions with some pedagogic feedback. Try to answer the following question. How much did the US GDP change in 2023, given the following facts:
    - First Republic Bank shareholders lost 2 billion dollars on March 16 due to the bank's collapse.
    - Microsoft decided to invest 10 billion dollars in the development of ChatGPT (OpenAI).
    - Amazon bought 100% of the capital of the company One Medical on February 22, 2023 for 4 billion dollars.<br>

    **a)** $−2$ billion $+10$ billion $+4$ billion.<br>
    **b)** $+4$ billion.<br>
    **c)** $+10$ billion $+4$ billion.<br>
    **d)** $+10$ billion.<br>
    **e)** None of the above.
    """)
    return


@app.cell(hide_code=True)
def _(feedback_q10, mo, q10_shaded):
    mo.hstack([q10_shaded, feedback_q10], justify="center", gap=6)
    return


@app.cell(hide_code=True)
def _(mo):
    # Display the radio button with options
    q10 = mo.ui.radio(
        options=["a", "b", "c", "d", "e"],
        inline=True, 
        label="***Please select an answer*** 👉 $~~$"
    )

    # Marimo objects automatically convert to HTML when placed in an f-string inside mo.Html
    q10_shaded = mo.Html(f"""
        <div style="position: relative; display: inline-block; padding: 10px;">
            {q10}
            <div style="
                position: absolute; 
                top: 0; left: 0; 
                width: 100%; height: 100%; 
                /* Use white (255, 255, 255) for dark mode */
                background: rgba(100, 149, 237, 0.35);
                pointer-events: none; 
                border-radius: 20px;
                z-index: 5;
            "></div>
        </div>
    """)

    q10_shaded;
    return q10, q10_shaded


@app.cell(hide_code=True)
def _(mo, q10):
    if q10.value is None:
        feedback_q10 = mo.md("👀 *We are waiting for your choice.*")
    elif q10.value == "d":
        feedback_q10 = mo.md("🎉 **Correct!** Microsoft's investment in ChatGPT development counts as new investment, increasing GDP by 10 billion. The bank collapse is a mere financial loss (doesn't affect GDP directly), and Amazon's acquisition is a transfer of existing assets (not new production).")
    else:
        feedback_q10 = mo.md("👎 **Wrong answer.** Try again. **But it is better to study first!** 😡")

    feedback_q10;
    return (feedback_q10,)


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    👉 **Exercise 8B: demand vs supply in a market economy**

    As you have studied in introductory microeconomics, price adjustments are the mechanism that equilibrates markets. The following figure illustrates the interaction between supply and demand in a market economy that leads to such adjustments.

    What kind of information can we get by looking at points 1 and 1'? And what happens in the case of points 2 and 2'?
    """)
    return


@app.cell(hide_code=True)
def _(go, np):
    # Supply and demand plot
    Q_market = np.arange(0.0, 25.0, 0.05)
    P_supply = 0.0 + 0.75 * Q_market
    P_demand = 18.0 - 0.75 * Q_market

    fig_market = go.Figure()

    fig_market.add_trace(go.Scatter(
        x=Q_market, y=P_demand, mode='lines',
        line=dict(color='maroon', width=2.2),
        name='Qᵈ (Demand)'
    ))

    fig_market.add_trace(go.Scatter(
        x=Q_market, y=P_supply, mode='lines',
        line=dict(color='Blue', width=2.2),
        name='Qˢ (Supply)'
    ))

    # Points at price 8.25
    fig_market.add_trace(go.Scatter(
        x=[11.0, 13.0], y=[8.25, 8.25],
        mode='markers+text', marker=dict(size=11, color='purple'),
        text=['1', "1'"], textposition='top center',
        textfont=dict(size=16, color='black'),
        showlegend=False, hoverinfo='skip'
    ))

    # Points at price 9.75
    fig_market.add_trace(go.Scatter(
        x=[11.0, 13.0], y=[9.75, 9.75],
        mode='markers+text', marker=dict(size=11, color='darkorange'),
        text=['2', "2'"], textposition='top center',
        textfont=dict(size=16, color='black'),
        showlegend=False, hoverinfo='skip'
    ))

    fig_market.update_layout(
        height=450,
        title_text='Demand (Qᵈ) versus supply (Qˢ)',
        title_x=0.5,
        hovermode='y unified',
        xaxis=dict(title='Quantity', range=[9.0, 15.0]),
        yaxis=dict(title='Price', range=[6.5, 11.5]),
        margin=dict(l=70, r=60, t=70, b=60)
    )

    fig_market
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    /// hint |Answer: Exercise 8B

    Points 1 and 1' tell us that for the price level of 8.25, there is excess demand (quantity demanded exceeds quantity supplied).

    Points 2 and 2' tell us that for the price level of 9.75, there is excess supply (quantity supplied exceeds quantity demanded).

    The price will be stable when there is equilibrium (where supply equals demand).
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ---
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ## 6. Downloading and plotting data
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    To download and work with data files, we use the package named as `pandas`. A CSV file is the most used format to deal with data and it means "comma separated values". To download the data file attached to this notebook – its name is <kbd>ECB Data Inflation.csv</kbd> – we should do as exemplified below. This data file contains the inflation rate for the EuroZone as a whole and four individual countries. This data was obtained from here.
    """)
    return


@app.cell
def _(pd, preview):
    inflation = pd.read_csv("ECB Data Inflation.csv")
    preview(inflation)
    return (inflation,)


@app.cell
def _(inflation):
    # Summary statistics
    inflation.describe()
    return


@app.cell
def _(inflation, px):
    # Plot single column
    fig5 = px.line(x=inflation.Months, y= inflation.Portugal,  title = "Inflation in Portugal")
    fig5.update_layout(height=500, 
                       xaxis_title = "Months", yaxis_title = "Percentage points",
                       margin=dict(l=60, r=60, t=70, b=60))
    fig5
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    To plot more than one column, we can create a vector [...] to include as many columns has we wish. In the case below we plot the columns associated with Germany and Portugal.
    """)
    return


@app.cell
def _(inflation, px):
    fig12 = px.line(x=inflation.Months, y=[inflation.Portugal , inflation.Germany], title="Inflation in Germany and Portugal", color_discrete_sequence=["#1919c9", "maroon"]) #1919c9 #0248ff

    fig12.update_traces(mode="lines", hovertemplate=None)

    fig12.update_layout(height = 500, hovermode="x", xaxis_title="Months", yaxis_title="Percentage points", title_x=0.5, margin=dict(l=70, r=70, t=70, b=60), legend_title=None)

    fig12.update_traces({'name': 'Portugal'}, selector={'name': 'wide_variable_0'})
    fig12.update_traces({'name': 'Germany'}, selector={'name': 'wide_variable_1'})
    fig12
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    The previous plot was made with Plotly Express <kbd>px</kbd>, which is the quickest way to perform plotting tasks. There is an alternative way of plotting the previous figure using the Plotly Graph Objects <kbd>go</kbd>. This new process provides more functionality for plotting, but it usually takes longer to write the code to produce the plots. The following plot was done using the `go` wrapper instead of `px`.
    """)
    return


@app.cell
def _(go, inflation):
    # Plot multiple columns
    fig6 = go.Figure()
    fig6.add_trace(go.Scatter(
        x=inflation.Months,
        y=inflation.Germany,
        mode='lines',
        name='Germany'
    ))
    fig6.add_trace(go.Scatter(
        x=inflation.Months,
        y=inflation.Portugal,
        mode='lines',
        name='Portugal'
    ))
    fig6.update_layout(
        height=500,
        title_text='Inflation in Germany and Portugal',
        title_x=0.5,
        hovermode='x',
        yaxis_title='Percentage points',
        xaxis_title='Monthly observations coco',
        margin=dict(l=70, r=60, t=70, b=60)
    )
    fig6
    return (fig6,)


@app.cell
def _():
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    👉 **Exercise 9**

    We can also save one plot as a "png" file, a "pdf" file, or an "svg" file (among many other formats). These files can be inserted later into another document (like a PowerPoint or a Word file). Their graphical quality will be much higher than by using the archaic "screen-capture" functionality. In the next cell, save "fig6" as an svg, a pdf, and a png file by copying and running the code <kbd>fig6.write_image("inflation_plot.svg")</kbd>. They will be automatically saved in the same folder where this notebook is located.
    """)
    return


@app.cell
def _(fig6):
    fig6.write_image("inflation_plot.svg")
    return


@app.cell
def _(fig6):
    fig6.write_image("inflation_plot.pdf")
    return


@app.cell
def _(fig6):
    fig6.write_image("inflation_plot.png")
    return


@app.cell(hide_code=True)
def _(go, inflation):
    fig7 = go.Figure()

        # Add each country trace individually
    fig7.add_trace(go.Scatter(
            x=inflation.Months,
            y=inflation.EuroArea,
            mode='markers+lines',
            marker=dict(size=4, color='blue'),
            line=dict(width=0.3, color='blue'),
            name='EuroArea'
        ))

    fig7.add_trace(go.Scatter(
            x=inflation.Months,
            y=inflation.Portugal,
            mode='markers+lines',
            marker=dict(size=4, color='maroon'),
            line=dict(width=0.3, color='maroon'),
            name='Portugal'
        ))

    fig7.add_trace(go.Scatter(
            x=inflation.Months,
            y=inflation.Germany,
            mode='markers+lines',
            marker=dict(size=4, color='cyan'),
            line=dict(width=0.3, color='cyan'),
            name='Germany'
        ))

    fig7.add_trace(go.Scatter(
            x=inflation.Months,
            y=inflation.Spain,
            mode='markers+lines',
            marker=dict(size=4, color='darkorange'),
            line=dict(width=0.3, color='darkorange'),
            name='Spain'
        ))

    fig7.add_trace(go.Scatter(
            x=inflation.Months,
            y=inflation.France,
            mode='markers+lines',
            marker=dict(size=4, color='purple'),
            line=dict(width=0.3, color='purple'),
            name='France'
        ))

    fig7.update_layout(
            height=500,
            title_text='Inflation in the EuroArea and four European countries',
            title_x=0.5,
            hovermode='x',
            xaxis_range=["1996-10-1", "2024-06-1"],
            yaxis_title='Percentage points',
            margin=dict(l=70, r=60, t=70, b=60)
        )

    fig7
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    Notice that there is a shorter way to produce the same plot as above: use a <kbd>for loop</kbd> for countries and colors. See the following figure.
    """)
    return


@app.cell
def _(go, inflation):
    # Comprehensive inflation plot

    fig7a = go.Figure()

    colors7a = ["blue", "maroon", "cyan", "darkorange", "teal"]
    countries7a = ['EuroArea', 'Portugal', 'Germany', 'Spain', 'France']

    for i, country in enumerate(countries7a):
        fig7a.add_trace(go.Scatter(
                x=inflation.Months,
                y=inflation[country],
                mode='markers+lines',
                marker=dict(size=4, color=colors7a[i]),
                line=dict(width=0.3, color=colors7a[i]),
                name=country
            ))

    fig7a.update_layout(
            height=500,
            title_text='Inflation in the EuroZone and Portugal',
            title_x=0.5,
            hovermode='x',
            xaxis_range=["1996-10-1", "2024-06-1"],
            yaxis_title='Percentage points',
            margin=dict(l=70, r=60, t=70, b=60)
        )

    fig7a
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    👉 **Exercise 9A**

    Create a simple plot of the following variable: <kbd>inflation.EuroArea  -  2</kbd>. That is, EuroArea inflation minus 2% inflation (2.5% inflation is the the inflation rate the European Central Bank (ECB) is asked to achieve). To produce that, apply the following code (see the plot below for more details):

    <kbd>fig8 = px.bar(x=inflation.Months, y=(inflation.EuroArea - 2)</kbd>

    where "px.bar" means that we want the data to be plotted as a bar, not as a line. The level of inflation the European Central Bank wants to see in the Euro Zone is 2%. Check whether the wishes of the ECB have been accomplished in the EU.
    """)
    return


@app.cell
def _(inflation, px):
    # Bar plot of inflation deviation from target
    fig8 = px.bar(x=inflation.Months, y=(inflation.EuroArea - 2), color_discrete_sequence=['darkred'] )
    fig8.update_layout(height=450, title_text='EuroArea Inflation Deviation from 2% Target', hovermode = "x")
    fig8
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    /// hint | Answer: Exercise 9A
    Between 2000 and mid-2007, the European central Bank (ECB) was very successful in maintaining inflation very close to the desired level: 2%. Since then, the ECB has shown tremendous difficulty maintaining inflation close to its desired level, and this is not good given the job that is expected from the ECB.
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ## 7. Using linear algebra
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    **Warning:** In computation, there is a crucial difference between a <kbd>,</kbd> and a <kbd>.</kbd> . In Python, we use <kbd>.</kbd> for decimal numbers: 1/2 = 0.5, NOT 1/2 = 0,5. If you mistakenly use a <kbd>,</kbd> to express decimal units, you will create a problem your computer does not know how to solve.
    """)
    return


@app.cell
def _(np):
    # Matrix operations
    A = np.array([[1, 2, 3], [4, 1, 6], [7, 8, 1]])
    A
    return (A,)


@app.cell
def _(A, np):
    # Matrix inverse
    A_inv = np.linalg.inv(A)
    A_inv
    return


@app.cell
def _(A):
    # Matrix transpose
    A_transpose = A.T
    A_transpose
    return


@app.cell
def _(A, np):
    # Matrix determinant
    A_det = np.linalg.det(A)
    A_det
    return


@app.cell
def _(A):
    # Matrix power
    A_squared = A @ A  # or np.linalg.matrix_power(A, 2)
    A_squared
    return


@app.cell
def _(np):
    # Random matrix
    Zak = np.random.randn(3, 3)
    Zak
    return (Zak,)


@app.cell
def _(A, Zak):
    # Matrix multiplication
    aple = A @ Zak
    aple
    return


@app.cell
def _(np):
    # Random integer matrix
    F = np.random.randint(1, 11, size=(5, 5))
    F
    return (F,)


@app.cell
def _(F):
    # Matrix indexing (Python uses 0-based indexing)
    F[0, 2]  # First row, third column
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    👉 **Exercise 10**

    Let us create a matrix of dimension (100×100), with all its elements as randomly generated numbers. Give this matrix the name Shrek and calculate its inverse.
    """)
    return


@app.cell
def _(np):
    # Large random matrix
    Shrek = np.random.randn(100, 100)
    Shrek
    return (Shrek,)


@app.cell
def _(Shrek, np):
    # Inverse of large matrix
    Shrek_inv = np.linalg.inv(Shrek)
    Shrek_inv
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    /// hint | Answer: Exercise 10

    The time it takes to compute the inverse of Shrek (a 100×100 matrix) is very fast with NumPy, typically under a millisecond on modern hardware.
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ---
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ## 8. Solving a complicated model
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    👉 **Exercise 11**

    Consider the following prototype of a macroeconomic model:

    $$\begin{aligned}
    & y_{1} = a + 0.5 y_{2} - 2.5 y_{3} \\
    & y_{2} = -2 y_{1} + 5 y_{3} + b \\
    & y_{3} = 10 y_{1} + 2 c \\
    & a = 10 \, , \, b=2 \, , \, c=5
    \end{aligned}$$

    We can solve this system using the <kbd>fsolve</kbd> function from the `scipy` library in Python. The first thing we should do is to pass the values of the variables with known values to the notebook:
    """)
    return


@app.cell
def _():
    # Model parameters
    a = 10
    b = 2
    c = 5
    p = [a , b , c] 
    return (p,)


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    /// tip | Steps to solve this exercise
    First, we rewrite the system as a homogeneous system (zeros on the left-hand side):

    $$\begin{aligned}
    &0 = a + 0.5 y_2 - 2.5 y_3 - y_1 \\
    &0 = -2 y_1 + 5 y_3 + b - y_2 \\
    &0 = 10 y_1 + 2c - y_3
    \end{aligned}$$

    Then we have to write down our problem according to the syntax of `scipy.optimize` (**see the image below for a graphical detailed description**)
    - The keyword <kbd>def</kbd> tells the computer that you are going to define your problem with a set of elements that will come next
    - Give a name to your problem: let us name it as <kbd>pumbaa</kbd>
    - Define the variables in our problem: <kbd>pumbaa(v,p)</kbd>
        -  <kbd>v</kbd> represents the set of **endogenous** variables (the letter used here `v` is arbitrary)
        -  <kbd>p</kbd> represents the set of **exogenous** variables (the letter `z` used here is arbitrary)
        -  <kbd>y1 , y2 , y3 = v</kbd> is the (unpacked) set of endogenous variables in the current problem (the order is arbitrary)
        -   <kbd>a , b , c = p</kbd> is the (unpacked) set of exogenous variables in the current problem (the order is arbitrary)
    - Define the equations in your problem:

    $$\begin{aligned}
    E = & \ [ \ a + 0.5 y_2 - 2.5 y_3 - y_1 \ , \\
    & \ \ \ 2 y_1 + 5 y_3 + b - y_2 \ , \\
    & \ \ \ 10 y_1 + 2c - y_3 \ ]
    \end{aligned}$$

    Finally, we should compute the numerical solution to our problem according to the syntax of `scipy.optimize` (see the image below for details about this last step).
    """)
    return


@app.cell
def _(fsolve, p):
    # Define our proble: equations (E), variables (v), and parameters (p)
    def pumbaa(v, p):
        y1, y2, y3 = v
        a, b , c = p 

        E = [ a + 0.5*y2 - 2.5*y3 - y1 , 
              - 2*y1 + 5*y3 + b - y2 , 
              10*y1 + 2*c -y3 ]

        return E

    solution_pumbaa = fsolve( pumbaa, [1.0, 1.0, 1.0] , args=(p) )
    solution_pumbaa
    return


@app.cell
def _(mo):
    mo.md(r"""
    <div style="display: flex; justify-content: left; align-items: left;">
        <img
    		src="http://modernmacro.org/resources/Vivaldo/figures/Marimo/fsolve_pumbaa.svg"
    	alt="Centered Image" style="width: 80%; height: auto;">
    </div>
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    /// hint | Answer:Exercise 11

    The solution is given by:  y1 = 5.5 , y2 = 316 , y3 = 65
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ---
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ## Summary

    In this notebook, we covered:

    1. Basic Marimo notebook concepts and reactive programming
    2. Markdown formatting and mathematical expressions
    3. Arithmetic operations and variable assignments
    4. Interactive plotting with Plotly
    5. Using sliders and interactive widgets
    6. Working with data using pandas
    7. Linear algebra with NumPy
    8. Solving systems of equations with scipy

    Marimo notebooks are reactive, meaning cells automatically re-run when their dependencies change. This ensures your notebook is always in a consistent state!
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ## Auxiliary cells (please do not change them)
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ### Functions
    """)
    return


@app.cell(hide_code=True)
def _(pd):
    # Making data frames looking like in Julia and Pluto (first 5 rows + last row)
    def preview(df, n=5):
        if len(df) <= n + 1:
            return df

        ellipsis = pd.DataFrame(
            [["..."] * df.shape[1]],
            columns=df.columns,
            index=["…"],
        )

        return pd.concat([
            df.head(n),
            ellipsis,
            df.tail(1),
        ])
    return (preview,)


@app.cell(hide_code=True)
def _(mo):
    # Global CSS for kbd elements throughout the notebook
    mo.Html("""
        <style>
            kbd {
                background-color: #505050 !important;
                color: #fff !important;
                padding: 3px 6px;
                border-radius: 4px;
                font-family: monospace;
                font-size: 0.9em;
                border: 0px solid #666;
            }
        </style>
    """)
    return


@app.cell(hide_code=True)
def _():
    #background-color: #546ccc !important; # A color that looks goog on white theme;
    return


if __name__ == "__main__":
    app.run()
